IF EXISTS ( SELECT * 
            FROM   sysobjects 
            WHERE  id = object_id(N'CPJ_Moratoria_Generar') 
                   and OBJECTPROPERTY(id, N'IsProcedure') = 1 )
BEGIN
    DROP PROCEDURE CPJ_Moratoria_Generar
END
GO

CREATE PROCEDURE [dbo].[CPJ_Moratoria_Generar] @pJson VARCHAR(8000)
AS
SET NOCOUNT ON

	/***************************************************************************************************

	Procedimiento:      dbo.CPJ_Moratoria_Generar
	Fecha de Creacion:  2023-07-07
	Creado por:         Florencia Gonzalez
	Descripcion:        Procedimiento encargado de circuito de moratoria (generacion de contrato)
	Llamado por:        [schema.usp_ProcThatCallsThis]
						[Application Name]
						[Job]
						[PLC/Interface]
	Tablas afectadas:   [Contratos],[Clientes],[Cbtes],[Cbtes_Items],[Clientes_CtaCte],[Cbtes_Aplica]
	Usado en:           Socios, Adhesion a Moratoria
	Parametro(s):       @pJson - Json con parametros FlagUsarTrans, ModoDebug, Id_Tipo_Con, Id_Cliente
	Forma de uso:       EXEC dbo.CPJ_Moratoria_Generar {"FlagUsarTrans":"0","ModoDebug":"1","Id_Tipo_Con":"51","Id_Cliente":"2716"}
	Notas adicionales:  

	ORDEN y GUIA DE EJECUCION:
	1) Generacion de contrato
		- Se genera el contrato del tipo seleccionado
		- Como Fecha Desde y Fecha Hasta se utilizan las del mes corriente

	2) Generacion de productos en contrato
		- Para el producto de Moratoria/Planes de pago se recalcula con el precio actual (utilizando la lista de precio del cliente)
			de acuerdo al valor del parametro Flag_Valor_Cuota_Actual del tipo de contrato
		- En caso de no recalcular el precio de CS, entonces se divide el total de la deuda de CS por el total de cuotas a generar
			(segun lo devuelto por la funcion CFUJ_Moratoria_Cbtes_Cant_Cuotas_Generar)
		- El producto de tipo Gastos Administrativos siempre se calcula a precio actual y cada cbte creado tiene 1 solo item de este producto

	3) Calculo de intereses
		- Se utiliza el procedimiento CPJ_Cbtes_Interes_Calc para calcular el interes de cada transaccion de CS pendiente
		- Los intereses se suman en la seccion de **Generacion de comprobante de deuda** a los nuevos cbtes. creados
		- Cada cbte creado tiene 1 solo item de este producto (si el contrato esta configurado con el producto de intereses)
		- Si se configura el tipo de contrato para que genere solo 1 comprobante, el producto de interes lleva el total de los intereses calculados
		- Si se configura el tipo de contrato para que genere 1 comprobante por cantidad de cbtes a generar (segun CFUJ_Moratoria_Cbtes_Cant_Cuotas_Generar)
			entonces por cada cbte habra un item correspondiente a los intereses y el importe del mismo sera el total de los intereses dividido
			por la cantidad de cbtes a generar

	4) Generacion de Cbte de deuda
		- Si se configura el tipo de contrato con los valores T (Abona por Unica Vez) o D (Genera un solo cbte de deuda pero no abona automaticamente)
			entonces se generara un solo tipo de cbte por el total de la deuda desglosado en el total de cbtes a generar (segun lo devuelto por la 
			funcion CFUJ_Moratoria_Cbtes_Cant_Cuotas_Generar), mas los productos de Gastos Admin. e Intereses (si fueron configurados)
		- Si se configura el tipo de contrato con los valores M (Desglosa deuda) se generaran el total de cbtes (segun lo devuelto por la funcion 
		CFUJ_Moratoria_Cbtes_Cant_Cuotas_Generar), cada uno con un item de Moratoria (y uno de Gastos Admin. e Intereses si fueron configurados)

	5) Cancelacion de deuda anterior
		- Se genera un recibo pago a cuenta (importe en $0) aplicando todas las transacciones pendientes de CS y se aplica con una NC (la configurada en el tipo de contrato
			seleccionado) por el total de la deuda, cancelando las transacciones pendientes y la NC todas en un mismo recibo.
		- Para la caja a utilizar en el recibo se consulta el parametro Caj_Per_Val_Rec y para el tipo de recibo a utilizar se consulta el parametro MORATORIA_Rec

	****************************************************************************************************

	SUMARIO DE CAMBIOS
	FECHA(yyyy-mm-dd)    Autor              Comentarios
	------------------- ------------------- ------------------------------------------------------------
	2023-07-17			Florencia G.		Se agrega parametro de Cond. de Pago segun lo configurado
											en contrato
    2025-04-15			Tomas C.			Se agrega parametro de Contrato existente e Id Contrato, si
											existe usa el contrato indicado por el parametro
	***************************************************************************************************/

	-------------------------------------------- VARIABLES RECIBIDAS POR PARAM --------------------------------------------

	DECLARE @pFlagUsarTrans			TINYINT
	DECLARE @pModoDebug				TINYINT
	DECLARE @pId_Tipo_Con			SMALLINT
	DECLARE @pId_Cliente			INT
	DECLARE @pId_Tarjeta			VARCHAR(10)
	DECLARE @pTar_Nro				VARCHAR(50)
	DECLARE @pOmitir_Resultado_OK	TINYINT
	-- TC.20250401
	DECLARE @pId_Contrato				INT
	DECLARE @pFlag_Contrato_Existente	TINYINT

	SET @pFlagUsarTrans				= ISNULL(CONVERT(TINYINT, dbo.CF_JSON_VALUE(@pJson,'FlagUsarTrans')), 0)
	SET @pModoDebug					= ISNULL(CONVERT(TINYINT, dbo.CF_JSON_VALUE(@pJson,'ModoDebug')), 0)
	SET @pId_Tipo_Con				= ISNULL(CONVERT(SMALLINT, dbo.CF_JSON_VALUE(@pJson,'Id_Tipo_Con')), 0)
	SET @pId_Cliente				= ISNULL(CONVERT(INT, dbo.CF_JSON_VALUE(@pJson,'Id_Cliente')), 0)
	SET @pId_Tarjeta				= ISNULL(CONVERT(VARCHAR(10), dbo.CF_JSON_VALUE(@pJson,'Id_Tarjeta')), '')
	SET @pTar_Nro				    = ISNULL(CONVERT(VARCHAR(50), dbo.CF_JSON_VALUE(@pJson,'Tar_Nro')), '')
	SET @pOmitir_Resultado_OK		= ISNULL(CONVERT(TINYINT, dbo.CF_JSON_VALUE(@pJson,'Omitir_Resultado_OK')), 0)
	-- TC.20250401
	SET @pId_Contrato				= ISNULL(CONVERT(INT, dbo.CF_JSON_VALUE(@pJson,'Id_Contrato')), 0)
	SET @pFlag_Contrato_Existente	= ISNULL(CONVERT(TINYINT, dbo.CF_JSON_VALUE(@pJson,'Flag_Contrato_Existente')), 0)

	-------------------------------------------- FIN VARIABLES RECIBIDAS POR PARAM --------------------------------------------

	-------------------------------------------- VARIABLES LOCALES --------------------------------------------

	DECLARE @JSON_Parametros_Stored_Invocado	VARCHAR(8000)
	DECLARE @rtaMsjStored						VARCHAR(8000)
	DECLARE @Categoria							CHAR(1)
	DECLARE @Cancelacion_Tipo					CHAR(1)
	DECLARE @Id_Contrato						INT
	DECLARE @Flag_Deb_Auto						SMALLINT
	DECLARE @Fecha_Desde_Contrato				DATETIME
	DECLARE @Fecha_Hasta_Contrato				DATETIME
	DECLARE @Id_Producto_Item					VARCHAR(14)
	DECLARE @Id_Producto_GAdmin					VARCHAR(14)
	DECLARE @Id_Producto_Interes				VARCHAR(14)
	DECLARE @Recalcula_CS						TINYINT
	DECLARE @Precio								DECIMAL(20,5)
	DECLARE @CantCuotas							INT
	DECLARE @Id_Trans_Generada					INT
	DECLARE @Id_Trans_NC						INT
	DECLARE	@Id_Trans_ND						INT
	DECLARE @Id_Trans_Rec						INT
	DECLARE @Id_Trans_Intereses					INT
	DECLARE @Total_Intereses					DECIMAL(20,2)
	DECLARE @Deuda_Ant							DECIMAL(20,2)
	DECLARE @Tipo_NC							CHAR(4)
	DECLARE @Tipo_ND							CHAR(4)
	DECLARE @Id_Lista_Precio_Cbtes				INT
	DECLARE @Id_Caja							INT
	DECLARE @Tipo_Rec							CHAR(4)
	DECLARE @Mensaje							VARCHAR(100)
	DECLARE @UltError							INT
	DECLARE @Segundos							INT		
	DECLARE @Items								SMALLINT
	DECLARE @Cond_Pago							TINYINT
	DECLARE @Id_Producto_CS						VARCHAR(14)
	DECLARE @Id_Grupo_Cont						INT
	DECLARE @Ult_Cuota_Paga						DATETIME
	DECLARE @Id_Tipo_Cli   					    INT
	DECLARE @Mes             					DATETIME
	DECLARE @Cant_Cuotas_Finan                  SMALLINT
	DECLARE @Fecha_Vto							DATETIME
	DECLARE @Fecha_Desde						DATETIME
	DECLARE @Cant_Deuda							INT
	DECLARE @Obs								VARCHAR(5000)
	DECLARE @Valor_Cuota						DECIMAL(20,2)
	DECLARE @Tipo_Cont_CS						INT
	DECLARE @CANT								INT
	DECLARE @Id_Contrato_Ant					INT

	DECLARE @Respuesta_ TABLE (
								ErrorNumber		INT,
								ErrorSeverity	INT,
								ErrorState		INT,
								ErrorProcedure  VARCHAR(1000),
								ErrorLine		INT,
								ErrorMessage	VARCHAR(1000)
							  )


	DECLARE @Trans_CS_Pendientes TABLE (
										  Id_Trans INT,
										  Fecha_QA DATETIME,
										  Imp_Final DECIMAL(15,5),
										  Item	   SMALLINT NOT NULL IDENTITY(1,1)
									   )

	DECLARE @Respuesta_Calc_Intereses TABLE (
											   Id_Trans		INT,
											   Recargo		DECIMAL(20,2)
											)

	SELECT	@Categoria			= ISNULL(Categoria,'')
			,@Cancelacion_Tipo	= ISNULL(Cancelacion_Tipo,'T')
			,@Recalcula_CS		= ISNULL(Flag_Valor_Cuota_Actual,0)
			,@Flag_Deb_Auto		= ISNULL(Flag_Deb_Auto,0)
			,@Id_Producto_Item	= ISNULL(Id_Producto_Item,'')
			,@Id_Producto_GAdmin	= ISNULL(Id_Producto_Gast_ADM,'')
			,@Id_Producto_Interes	= ISNULL(Id_Producto_Interes,'')
			,@Cond_Pago				= ISNULL(Id_Cond_Pago,0)
			,@Id_Grupo_Cont			= ISNULL(Cant_Cuotas,1)
			,@Cant_Cuotas_Finan     = ISNULL(Cant_Cuotas,1)
			 FROM Contratos_Tipos WHERE Id_Tipo_Con = @pId_Tipo_Con
	SET @Items						= 0
	SET @Id_Producto_CS				= ISNULL(dbo.CF_xParam('Prod_Cuota_Social'),'CS')
	SET @Fecha_Desde_Contrato		= DATEADD(DAY,1,EOMONTH(GETDATE(),-1))
	SET @Fecha_Hasta_Contrato		= DATEADD(MONTH,@Cant_Cuotas_Finan,@Fecha_Desde_Contrato)
	SET @Id_Tipo_Cli				= (SELECT Id_Tipo_Cli FROM Clientes WHERE Id_Cliente = @pId_Cliente)
	SET @Ult_Cuota_Paga				= (SELECT ISNULL(Ult_Cuota_Paga,Fecha_Alta) FROM Clientes WHERE Id_Cliente = @pId_Cliente)
	SET @Fecha_Desde				= DATEADD(MONTH,1,@Ult_Cuota_Paga)
	SET @Tipo_Cont_CS				= ISNULL(dbo.CF_xParam('Tipo_Cont_CS'),0)

	SET @JSON_Parametros_Stored_Invocado	= '{' + dbo.CF_Json_ParsearVariable('Id_Cliente', @pId_Cliente, 'INT') + '}'
	SET @CantCuotas							= dbo.CFUJ_Moratoria_Cant_Cuotas_Adeudadas(@JSON_Parametros_Stored_Invocado)
	SET @Deuda_Ant							= (SELECT dbo.CFUJ_Moratoria_Deuda_Generada(@JSON_Parametros_Stored_Invocado))
	IF(@Recalcula_CS = 1)
	BEGIN
		IF(@pModoDebug = 1)
		BEGIN
			PRINT 'El contrato recalcula valor de CS'
		END
		SET @Precio	= dbo.CF_Produc_Precio(dbo.CF_xParam('Prod_Cuota_Social'), @pId_Cliente, dbo.CF_FOMONTH(GETDATE()), 0, 0,  'B')
	END
	ELSE
	BEGIN
		SET @Precio = (SELECT dbo.CFUJ_Moratoria_Deuda(@JSON_Parametros_Stored_Invocado))
	END
	SET @Valor_Cuota = (@Precio / @Cant_Cuotas_Finan)


	SET @Cant_Deuda = (SELECT DATEDIFF(MONTH,@Fecha_Desde,GETDATE()))
	SET @Obs = 'Meses Adeudados: '+ CONVERT(VARCHAR,@Fecha_Desde,103)  +' al '+  CONVERT(VARCHAR,GETDATE(),103) + CHAR(13) + CHAR(10) +
				'Deuda Anterior Generada:  '+ CONVERT(VARCHAR(15),@Deuda_Ant) + CHAR(13) + CHAR(10) +
				'Deuda Recalculada:  '+ CONVERT(VARCHAR(15),CONVERT(DECIMAL(20,2),@Precio)) + CHAR(13) + CHAR(10) +
				'Cantidad de Pagos:  '+ CONVERT(VARCHAR(15),@Cant_Cuotas_Finan) + CHAR(13) + CHAR(10) +
				'Valor de cada pago:  '+ CONVERT(VARCHAR(15),@Valor_Cuota)

	IF(@pModoDebug = 1)
	BEGIN
		PRINT '@Fecha_Desde_Contrato: ' + CONVERT(VARCHAR(10),@Fecha_Desde_Contrato,112)
		PRINT '@Fecha_Hasta_Contrato: ' + CONVERT(VARCHAR(10),@Fecha_Hasta_Contrato,112)
		PRINT '@Fecha_Desde: ' + CONVERT(VARCHAR(15),@Fecha_Desde,112)
		PRINT '@Cant_Deuda: ' + CONVERT(VARCHAR(15),@Cant_Deuda) 
	END

	IF((SELECT Id_Iva FROM Clientes WHERE Id_Cliente = @pId_Cliente) = dbo.CF_xParam('Iva_RI'))
	BEGIN
		SET @Tipo_NC = ISNULL((SELECT Id_Tipo_Cbte_NC_RI FROM Contratos_Tipos WHERE Id_Tipo_Con = @pId_Tipo_Con),0)
	END
	ELSE
	BEGIN
		SET @Tipo_NC = ISNULL((SELECT Id_Tipo_Cbte_NC FROM Contratos_Tipos WHERE Id_Tipo_Con = @pId_Tipo_Con),0)
	END

	SET @Tipo_ND = ISNULL((SELECT Id_Tipo_Cbte FROM Contratos_Tipos WHERE Id_Tipo_Con = @pId_Tipo_Con),'')

	IF(@CantCuotas = 0)
	BEGIN
		PRINT 'No hay cuotas pendientes del socio'
		RETURN 0
	END

	INSERT INTO @Trans_CS_Pendientes
			SELECT I.Id_Trans, I.Fecha_QA, I.Imp_Final
			FROM Cbtes_Items I
				LEFT JOIN Cbtes C ON I.Id_Trans = C.Id_Trans
				LEFT JOIN Cbtes_Tipos CT ON C.Id_Tipo_Cbte = CT.Id_Tipo_Cbte
			WHERE  I.Id_Cliente = @pId_Cliente
				AND I.Id_Producto = @Id_Producto_CS
				AND C.Id_Estado_Cbte IN (1,4)
				AND C.Imp_Total > 0
				AND C.Fecha < GETDATE() + 365
				AND CT.Compromete_Factura IN (1,-1)

    IF @Flag_Deb_Auto = 1
	BEGIN
 	    SET @Cond_Pago = ISNULL(dbo.CF_xParam('Cond_Pag_Debito'),@Cond_Pago) 
	END
    ELSE
	BEGIN
	   SET @Cond_Pago = ISNULL(dbo.CF_xParam('Cond_Pago_CTACTE'),@Cond_Pago) 
	   SET @pId_Tarjeta = ''
	   SET @pTar_Nro = ''
    END

	-------------------------------------------- FIN VARIABLES LOCALES --------------------------------------------

IF @pFlagUsarTrans <> 0 AND @@TRANCOUNT > 0 
	SET @pFlagUsarTrans = 0

BEGIN TRY
	IF(@pModoDebug = 1)
	BEGIN
		PRINT 'Ingreso a Stored Procedure: CPJ_Moratoria_Generar'
		PRINT 'Parametros:'
		PRINT 'pModoDebug: ' + CONVERT(VARCHAR(5), @pModoDebug) + '. pId_Tipo_Con: ' + CONVERT(VARCHAR(10), @pId_Tipo_Con)
	END

	IF @pFlagUsarTrans <> 0 
	BEGIN
		BEGIN TRAN
	END
	
/* Valido que no tenga un contrato activo ya antes*/



	SET @Id_Contrato_Ant = ISNULL((SELECT TOP 1 Id_Contrato FROM Contratos C, Contratos_Tipos T 
					WHERE C.Id_Cliente = @pId_Cliente AND C.Activo = 1 
				AND C.Id_Tipo_Con = T.Id_Tipo_Con
				AND T.Categoria IN ('M','P')
				AND Fecha_Desde <= GETDATE()
				AND Fecha_Hasta >= GETDATE()
				),0)

	IF(@Id_Contrato_Ant <> 0 AND @pFlag_Contrato_Existente = 0)
	BEGIN
		RAISERROR ('Error: Ya tiene un plan de pagos/moratoria activa', 16, 1) 
	END

/*************************************************************************/	
	
	
	

	/*Activo el cliente si estaba dado de baja y le coloco la categor├¡a actual*/
	UPDATE Clientes SET Activo = 1, Id_Tipo_cli = Dbo.SF_Cli_Categ_Mes(@pId_Cliente,Getdate()) 
			,Fecha_Baja = NULL
			,Id_Motivo_Est = 0
			WHERE Id_Cliente = @pId_Cliente

	/*Si eligio debito auto se pasa al contrato de cuota social*/
	IF @Tipo_Cont_CS <> 0 AND  @Flag_Deb_Auto = 1 AND @pId_Tarjeta <> '' AND @pTar_Nro	<> ''
		UPDATE Contratos SET Id_Tarjeta = @pId_Tarjeta , Tar_Nro = @pTar_Nro WHERE Id_Cliente = @pId_Cliente AND Id_Tipo_Con = @Tipo_Cont_CS AND Activo = 1

		

	-------------------------------------------- GENERACION DE CONTRATO --------------------------------------------

	IF(@pFlag_Contrato_Existente = 1 AND @pId_Contrato <> 0) -- TC.20250401 - Agrega la opcion si el contrato fue generado, toma este y lo actualiza
	BEGIN
		IF(@pModoDebug = 1)
		BEGIN
			PRINT 'Utiliza un contrato existente'
			PRINT 'Parametros:'
			PRINT '@pId_Contrato: ' + CAST(@pId_Contrato AS VARCHAR)
		END

		SET @Id_Contrato = @pId_Contrato
		
		UPDATE Contratos SET Fecha_Desde = @Fecha_Desde_Contrato, Fecha_Hasta = @Fecha_Hasta_Contrato, Observacion = @Obs WHERE Id_Contrato = @Id_Contrato

		IF(@pModoDebug = 1)
		BEGIN
			PRINT 'Actualiza del contrato existente: Fecha_Desde, Fecha_Hasta, Observacion'
		END
	END
	ELSE
	BEGIN
		SET @JSON_Parametros_Stored_Invocado =	'{' +
													dbo.CF_Json_ParsearVariable('FlagUsarTrans', @pFlagUsarTrans, 'TINYINT') + ',' +
													dbo.CF_Json_ParsearVariable('ModoDebug', @pModoDebug, 'TINYINT') + ',' +
													dbo.CF_Json_ParsearVariable('Id_Usuario', -1, 'SMALLINT') + ',' +
													dbo.CF_Json_ParsearVariable('Id_Tipo_Con', @pId_Tipo_Con, 'INT') + ',' +
													dbo.CF_Json_ParsearVariable('Fecha_Desde', @Fecha_Desde_Contrato, 'DATETIME') + ',' +
													dbo.CF_Json_ParsearVariable('Fecha_Hasta', @Fecha_Hasta_Contrato, 'DATETIME') + ',' +
													dbo.CF_Json_ParsearVariable('Id_Cliente', @pId_Cliente, 'INT') + ',' +
													dbo.CF_Json_ParsearVariable('Id_Grupo_Cont', '', 'CHAR(10)') + ',' +
													dbo.CF_Json_ParsearVariable('Id_SubGrupo_Cont', '', 'CHAR(10)') + 
													dbo.CF_Json_ParsearVariable('Observacion', @Obs, 'VARCHAR(5000)') + 
													dbo.CF_Json_ParsearVariable('Flag_Sistema', 1, 'TINYINT') + 
												'}'

		IF(@pModoDebug = 1)
		BEGIN
			PRINT 'Ingreso a Stored Procedure: CPJ_Contratos_Generar'
			PRINT 'Parametros:'
			PRINT '@JSON_Parametros_Stored_Invocado: ' + @JSON_Parametros_Stored_Invocado
		END

		INSERT INTO @Respuesta_
			EXEC @UltError = dbo.CPJ_Contratos_Generar @JSON_Parametros_Stored_Invocado

		IF(@UltError <> 0)
		BEGIN
			RAISERROR ('Error when calling CPJ_Contratos_Generar', 16, 1) 
		END


		SET @Id_Contrato = ISNULL(CONVERT(INT, dbo.CF_JSON_VALUE((SELECT ErrorMessage FROM @Respuesta_),'Id_Contrato')), 0)
		--UPDATE Contratos SET Id_Lista_Precio = 0 WHERE Id_Contrato = @Id_Contrato

		IF(@pModoDebug = 1)
		BEGIN
			PRINT 'Nuevo Contrato generado: ' + CONVERT(VARCHAR(25),@Id_Contrato)
		END
	END

	IF(@Flag_Deb_Auto = 1)
	BEGIN
		IF(@pId_Tarjeta <> '' AND @pTar_Nro <> '')
		BEGIN
			IF(@pModoDebug = 1)
			BEGIN
				PRINT 'El tipo de contrato permite debito automatico. Se carga la tarjeta codigo ' + @pId_Tarjeta + ' y el nro. de tarjeta ' + @pTar_Nro + ' en el contrato ' + CONVERT(VARCHAR(25),@Id_Contrato)
			END

			UPDATE Contratos SET Id_Tarjeta = @pId_Tarjeta, Tar_Nro = @pTar_Nro WHERE Id_Contrato = @Id_Contrato
		END
	END

	-------------------------------------------- FIN GENERACION DE CONTRATO -------------------------------------------
	
	------------------------------------------ GENERACION DE PROD EN CONTRATO ------------------------------------------

	IF(@pModoDebug = 1)
	BEGIN
		PRINT 'Se actualizan los productos del contrato con sus precios'
	END
	


	IF(@pModoDebug = 1)
	BEGIN
		PRINT 'Precio de Item de Moratoria/Plan de Pago Socio: ' + CONVERT(VARCHAR(25),@Precio)
	END

	INSERT INTO Contratos_Prod(Id_Contrato,Item,Id_Producto,Precio,Flag_Facturable,Cantidad,Cant_Cuotas)
		VALUES (@Id_Contrato,1,@Id_Producto_Item,@Precio,1,1,@Cant_Cuotas_Finan)

	IF(@Id_Producto_GAdmin <> '')
	BEGIN
		SET @Precio	= dbo.CF_Produc_Precio(@Id_Producto_GAdmin, @pId_Cliente, dbo.CF_FOMONTH(GETDATE()), @Id_Contrato, 0,  'B')

		IF(@pModoDebug = 1)
		BEGIN
			PRINT 'Precio de Gastos Admin del Socio: ' + CONVERT(VARCHAR(25),@Precio)
		END

		INSERT INTO Contratos_Prod(Id_Contrato,Item,Id_Producto,Precio,Flag_Facturable,Cantidad,Cant_Cuotas)
			VALUES (@Id_Contrato,2,@Id_Producto_GAdmin,@Precio,1,1,1)
	END

	------------------------------------------ FIN GENERACION DE PROD EN CONTRATO ------------------------------------------



	------------------------------------------ CANCELACION DE DEUDA PREVIA ------------------------------------------
    IF (@Deuda_Ant > 0)
    BEGIN
	/* GENERACION NC Y RECIBO CON CBTES PENDIENTES ANTERIORES A MORATORIA APLICADOS */

	IF(@pModoDebug = 1)
	BEGIN
		PRINT 'Genero NC por el total de la deuda de CS'
		PRINT 'Genero NC tipo: ' + @Tipo_NC
	END

	SET @JSON_Parametros_Stored_Invocado =	'{' +
													dbo.CF_Json_ParsearVariable('FlagUsarTrans', 0, 'TINYINT') + ',' +
													dbo.CF_Json_ParsearVariable('ModoDebug', @pModoDebug, 'TINYINT') + ',' +										
													dbo.CF_Json_ParsearVariable('Id_Usuario', -1, 'SMALLINT') + ',' +
													dbo.CF_Json_ParsearVariable('Omitir_Resultado_OK', 1, 'TINYINT') + ',' +
													dbo.CF_Json_ParsearVariable('Lote', '', 'VARCHAR(20)') + ',' +
													dbo.CF_Json_ParsearVariable('Fecha', GETDATE(), 'DATE') + ',' +
													dbo.CF_Json_ParsearVariable('FechaVence', GETDATE(), 'DATE') + ',' +
													dbo.CF_Json_ParsearVariable('Id_Cliente', @pId_Cliente, 'INT') + ',' +
													dbo.CF_Json_ParsearVariable('Id_Tipo_Cbte', @Tipo_NC, 'CHAR(4)') + ',' +
													dbo.CF_Json_ParsearVariable('Id_Cond_Pago', @Cond_Pago, 'SMALLINT') + ',' +
													dbo.CF_Json_ParsearVariable('Id_Moneda', 'PES', 'CHAR(3)') + ',' +
													dbo.CF_Json_ParsearVariable('Id_Lista_Precio', @Id_Lista_Precio_Cbtes, 'SMALLINT') + ',' +
													dbo.CF_Json_ParsearVariable('Id_Contrato', @Id_Contrato, 'INT') +
													'}'

	IF(@pModoDebug = 1)
	BEGIN
		PRINT 'Ingreso a Stored Procedure: CPJ_Cbtes_Fac_Generar'
		PRINT 'Parametros:'
		PRINT '@JSON_Parametros_Stored_Invocado: ' + @JSON_Parametros_Stored_Invocado
	END														
													
	EXEC @UltError = dbo.CPJ_Cbtes_Fac_Generar @JSON_Parametros_Stored_Invocado

	IF(@UltError <> 0)
	BEGIN
		RAISERROR ('Error when calling CPJ_Cbtes_Fac_Generar NC', 16, 1) 
	END

	

	SELECT TOP 1 @Id_Trans_NC = Id_Trans FROM Cbtes WHERE Id_Contrato = @Id_Contrato AND Id_Tipo_Cbte = @Tipo_NC ORDER BY Id_Trans DESC

	IF(@pModoDebug = 1)
	BEGIN
		PRINT 'Transaccion de NC generada: ' + CONVERT(VARCHAR(25),@Id_Trans_NC)
	END



	IF(@pModoDebug = 1)
	BEGIN
		PRINT 'Genero item de Moratoria para NC por el total de ' + CONVERT(VARCHAR(30),@Precio)
	END

	SET @JSON_Parametros_Stored_Invocado =	'{' +
												dbo.CF_Json_ParsearVariable('FlagUsarTrans', 0, 'TINYINT') + ',' +
												dbo.CF_Json_ParsearVariable('ModoDebug', @pModoDebug, 'TINYINT') + ',' +										
												dbo.CF_Json_ParsearVariable('Id_Trans', @Id_Trans_NC, 'INT') + ',' +
												dbo.CF_Json_ParsearVariable('Omitir_Resultado_OK', 1, 'TINYINT') + ',' +
												dbo.CF_Json_ParsearVariable('Id_Producto', @Id_Producto_Item, 'VARCHAR(14)') + ',' +
												dbo.CF_Json_ParsearVariable('Cantidad', 1, 'DECIMAL(15,5)') + ',' +
												dbo.CF_Json_ParsearVariable('Fecha_QA', @Fecha_Desde_Contrato, 'DATETIME') + ',' +
												dbo.CF_Json_ParsearVariable('Fecha_Venc', @Fecha_Desde_Contrato, 'DATETIME') + ',' +
												dbo.CF_Json_ParsearVariable('Id_Cliente', @pId_Cliente, 'INT') + ',' +
												dbo.CF_Json_ParsearVariable('Lote', '', 'VARCHAR(20)') + ',' +
												dbo.CF_Json_ParsearVariable('Refacturar', 0, 'TINYINT') + ',' +
												dbo.CF_Json_ParsearVariable('Bonif', 0, 'DECIMAL(5,2)') + ',' +
												dbo.CF_Json_ParsearVariable('Precio_Grav', @Deuda_Ant, 'DECIMAL(15,5)') + ',' +
												dbo.CF_Json_ParsearVariable('Cantidad_Equiv', 0, 'DECIMAL(15,5)') + ',' +
												dbo.CF_Json_ParsearVariable('Precio_Exen', 0, 'DECIMAL(15,5)') + 
											'}'

	IF(@pModoDebug = 1)
	BEGIN
		PRINT 'Ingreso a Stored Procedure: CPJ_Cbtes_Fac_Generar_Item'
		PRINT 'Parametros:'
		PRINT '@JSON_Parametros_Stored_Invocado: ' + @JSON_Parametros_Stored_Invocado
	END			

	EXEC @UltError = dbo.CPJ_Cbtes_Fac_Generar_Item @JSON_Parametros_Stored_Invocado

	IF(@UltError <> 0)
	BEGIN
		RAISERROR ('Error when calling CPJ_Cbtes_Fac_Generar_Item', 16, 1) 
	END

	IF(@pModoDebug = 1)
	BEGIN
		PRINT 'Genero Cuentas Contables e ingreso de Clientes Cta Cte la NC '
	END

	SET @JSON_Parametros_Stored_Invocado =	'{' +
												dbo.CF_Json_ParsearVariable('FlagUsarTrans', 0, 'TINYINT') + ',' +
												dbo.CF_Json_ParsearVariable('ModoDebug', @pModoDebug, 'TINYINT') + ',' +
												dbo.CF_Json_ParsearVariable('Omitir_Result_OK', 1, 'TINYINT') + ',' +
												dbo.CF_Json_ParsearVariable('Id_Trans', @Id_Trans_NC, 'INT') + 
											'}'
	
	IF(@pModoDebug = 1)
	BEGIN
		PRINT 'Ingreso a Stored Procedure: CPJ_Cbtes_Fac_Fin (3)'
		PRINT 'Parametros:'
		PRINT '@JSON_Parametros_Stored_Invocado: ' + @JSON_Parametros_Stored_Invocado
	END	

	EXEC @UltError = dbo.CPJ_Cbtes_Fac_Fin @JSON_Parametros_Stored_Invocado

	IF(@UltError <> 0)
	BEGIN
		RAISERROR ('Error when calling CPJ_Cbtes_Fac_Fin (3)', 16, 1) 
	END
	
	SET @Id_Caja = (SELECT dbo.CF_XParam('Caj_Per_Val_Rec'))
	SET @Tipo_Rec = (SELECT dbo.CF_XParam('MORATORIA_Rec'))

	IF(@pModoDebug = 1)
	BEGIN
		PRINT 'Genero Recibo por el total de deuda de CS'
		PRINT 'Genero Recibo tipo: ' + @Tipo_Rec + ' en el periodo de caja ' + CONVERT(VARCHAR(25),@Id_Caja)
	END

	SET @JSON_Parametros_Stored_Invocado =	'{' +
											dbo.CF_Json_ParsearVariable('FlagUsarTrans', 1, 'TINYINT') + ',' +
											dbo.CF_Json_ParsearVariable('ModoDebug', @pModoDebug, 'TINYINT') + ',' +
											dbo.CF_Json_ParsearVariable('Id_Cliente', @pId_Cliente, 'INT') + ',' +
											dbo.CF_Json_ParsearVariable('Id_Usuario', -1, 'SMALLINT') + ',' +
											dbo.CF_Json_ParsearVariable('Omitir_Resultado_OK', 1, 'TINYINT') + ',' +					
											dbo.CF_Json_ParsearVariable('Id_Periodo_Caja', @Id_Caja, 'INT') + ',' +
											dbo.CF_Json_ParsearVariable('Fecha', GETDATE(), 'DATETIME') + ',' +
											dbo.CF_Json_ParsearVariable('Id_Tipo_Cbte', @Tipo_Rec, 'CHAR(4)') + ',' +
											dbo.CF_Json_ParsearVariable('Id_Moneda', 'PES', 'CHAR(3)') + ',' +
											dbo.CF_Json_ParsearVariable('Lote', 'MORATORIA', 'VARCHAR(20)') + ',' +
											dbo.CF_Json_ParsearVariable('Id_Contrato', @Id_Contrato, 'INT') + ',' +
											dbo.CF_Json_ParsearVariable('Importe', 0, 'DECIMAL(20,2)') +
										'}'

	IF(@pModoDebug = 1)
	BEGIN
		PRINT 'Ingreso a Stored Procedure: CPJ_Cbtes_Rec_Generar'
		PRINT 'Parametros:'
		PRINT '@JSON_Parametros_Stored_Invocado: ' + @JSON_Parametros_Stored_Invocado
	END	
										
	EXEC @UltError = dbo.CPJ_Cbtes_Rec_Generar @JSON_Parametros_Stored_Invocado

	IF(@UltError <> 0)
	BEGIN
		RAISERROR ('Error when calling CPJ_Cbtes_Rec_Generar', 16, 1) 
	END

	SELECT TOP 1 @Id_Trans_Rec = Id_Trans 
		   FROM Cbtes 
		   WHERE Lote = 'MORATORIA' AND Id_Cliente = @pId_Cliente
		   ORDER BY Id_Trans DESC

	IF(@pModoDebug = 1)
	BEGIN
		PRINT 'Recibo realizado, transaccion: ' + CONVERT(VARCHAR(25),@Id_Trans_Rec)
	END

	SET @CANT = (SELECT COUNT(*) FROM @TRANS_CS_PENDIENTES)

	SET @Items = 1
	WHILE(@Items <= @CANT)
	BEGIN
		SELECT @Id_Trans_Generada = Id_Trans, @Precio = Imp_Final FROM @TRANS_CS_PENDIENTES WHERE Item = @Items

		IF(@pModoDebug = 1)
		BEGIN
			PRINT 'Se aplica Transaccion: ' + CONVERT(VARCHAR(25),@Id_Trans_Generada) + ' - cancelando importe: ' + CONVERT(VARCHAR(25),@Precio)
		END

		SET @JSON_Parametros_Stored_Invocado =	'{' +
										dbo.CF_Json_ParsearVariable('FlagUsarTrans', 0, 'TINYINT') + ',' +
										dbo.CF_Json_ParsearVariable('ModoDebug', @pModoDebug, 'TINYINT') + ',' +
										dbo.CF_Json_ParsearVariable('Id_Trans', @Id_Trans_Rec, 'INT') + ',' +
										dbo.CF_Json_ParsearVariable('Item', @Items, 'SMALLINT') + ',' +
										dbo.CF_Json_ParsearVariable('Omitir_Resultado_OK', 1, 'TINYINT') + ',' +					
										dbo.CF_Json_ParsearVariable('Id_Producto', @Id_Producto_Item, 'VARCHAR(20)') + ',' +
										dbo.CF_Json_ParsearVariable('Cantidad', 1, 'DECIMAL(15,5)') + ',' +
										dbo.CF_Json_ParsearVariable('Id_Trans_A_Cancelar', @Id_Trans_Generada, 'INT') + ',' +
										dbo.CF_Json_ParsearVariable('Importe_A_Cancelar', @Precio, 'DECIMAL(16,2)') +
									'}'
		IF(@pModoDebug = 1)
		BEGIN
			PRINT 'Ingreso a Stored Procedure: CPJ_Cbtes_Rec_Generar_Item (1)'
			PRINT 'Parametros:'
			PRINT '@JSON_Parametros_Stored_Invocado: ' + @JSON_Parametros_Stored_Invocado
		END	
										
		EXEC @UltError =  dbo.CPJ_Cbtes_Rec_Generar_Item @JSON_Parametros_Stored_Invocado
		
		IF(@UltError <> 0)
		BEGIN
			RAISERROR ('Error when calling CPJ_Cbtes_Rec_Generar_Item (1)', 16, 1) 
		END

		SET @Items = @Items + 1
	END
	
	SET @Precio = (SELECT SUM(Imp_Final) * -1 FROM @TRANS_CS_PENDIENTES)
	SET @Items += 1

	SET @JSON_Parametros_Stored_Invocado =	'{' +
												dbo.CF_Json_ParsearVariable('FlagUsarTrans', 0, 'TINYINT') + ',' +
												dbo.CF_Json_ParsearVariable('ModoDebug', @pModoDebug, 'TINYINT') + ',' +
												dbo.CF_Json_ParsearVariable('Id_Trans', @Id_Trans_Rec, 'INT') + ',' +
												dbo.CF_Json_ParsearVariable('Item', @Items, 'SMALLINT') + ',' +
												dbo.CF_Json_ParsearVariable('Omitir_Resultado_OK', 1, 'TINYINT') + ',' +					
												dbo.CF_Json_ParsearVariable('Id_Producto', @Id_Producto_Item, 'VARCHAR(20)') + ',' +
												dbo.CF_Json_ParsearVariable('Cantidad', 1, 'DECIMAL(15,5)') + ',' +
												dbo.CF_Json_ParsearVariable('Id_Trans_A_Cancelar', @Id_Trans_NC, 'INT') + ',' +
												dbo.CF_Json_ParsearVariable('Importe_A_Cancelar', @Precio, 'DECIMAL(16,2)') +
											'}'

	IF(@pModoDebug = 1)
	BEGIN
		PRINT 'Ingreso a Stored Procedure: CPJ_Cbtes_Rec_Generar_Item (2)'
		PRINT 'Parametros:'
		PRINT '@JSON_Parametros_Stored_Invocado: ' + @JSON_Parametros_Stored_Invocado
	END	
										
	EXEC @UltError = dbo.CPJ_Cbtes_Rec_Generar_Item @JSON_Parametros_Stored_Invocado

	IF(@UltError <> 0)
	BEGIN
		RAISERROR ('Error when calling CPJ_Cbtes_Rec_Generar_Item (2)', 16, 1) 
	END

	IF(@pModoDebug = 1)
	BEGIN
		PRINT 'Ingreso a Stored Procedure: CPJ_Cbtes_Rec_Fin'
		PRINT 'Parametros:'
		PRINT '@JSON_Parametros_Stored_Invocado: ' + @JSON_Parametros_Stored_Invocado
	END	
	
	SET @JSON_Parametros_Stored_Invocado =	'{' +
								dbo.CF_Json_ParsearVariable('FlagUsarTrans', 0, 'TINYINT') + ',' +
								dbo.CF_Json_ParsearVariable('ModoDebug', @pModoDebug, 'TINYINT') + ',' +
								dbo.CF_Json_ParsearVariable('Id_Trans', @Id_Trans_Rec, 'INT') + ',' +
								dbo.CF_Json_ParsearVariable('Flag_Renumerar', 1, 'TINYINT') + ',' +
								dbo.CF_Json_ParsearVariable('Omitir_Resultado_OK', 1, 'TINYINT') +
							'}'

	EXEC @UltError = dbo.CPJ_Cbtes_Rec_Fin @JSON_Parametros_Stored_Invocado

	IF(@UltError <> 0)
	BEGIN
		RAISERROR ('Error when calling CPJ_Cbtes_Rec_Fin', 16, 1) 
	END

	------------------------------------------ FIN CANCELACION DE DEUDA PREVIA ------------------------------------------
	END







	--RAISERROR ('Error when calling CPJ_Cbtes_Fac_Generar NC', 16, 1) --sacar






	------------------------------------------ CALCULO DE INTERESES ------------------------------------------

	IF(@Id_Producto_Interes <> '')
	BEGIN
		IF(@pModoDebug = 1)
		BEGIN
			PRINT 'El tipo de contrato genera intereses'
			PRINT 'Se ingresa al calculo de intereses para la deuda anterior'
		END

		PRINT 'Items ' + CONVERT(VARCHAR(25),@Items) + ' Cant_Cuotas ' + CONVERT(VARCHAR(25),@CantCuotas) 
		WHILE(@Items < @CantCuotas)
		BEGIN
			SET @Items = @Items + 1

			SELECT @Id_Trans_Intereses = Id_Trans FROM @Trans_CS_Pendientes WHERE Item = @Items

			SET @JSON_Parametros_Stored_Invocado = '{' +
														dbo.CF_Json_ParsearVariable('Id_Cliente', @pId_Cliente, 'INT') + ',' +
														dbo.CF_Json_ParsearVariable('Id_Unid_Negocio', '', 'VARCHAR(10)') + ',' +
														dbo.CF_Json_ParsearVariable('Criterio', 'T', 'VARCHAR(1)') + ',' +		
														dbo.CF_Json_ParsearVariable('Id_Trans', @Id_Trans_Intereses, 'INT') + 
													'}'

											
			INSERT INTO @Respuesta_Calc_Intereses
			EXEC dbo.CPJ_Cbtes_Interes_Calc @JSON_Parametros_Stored_Invocado
		END

		SET @Total_Intereses = (SELECT SUM(Recargo) FROM @Respuesta_Calc_Intereses)

		PRINT 'Total de intereses: ' + CONVERT(VARCHAR(25),@Total_Intereses)
	END

	------------------------------------------ FIN CALCULO DE INTERESES ------------------------------------------
	






	------------------------------------------ GENERACION ND CON LAS CUOTAS INCLUIDAS EN CERO  Y PRIMER CUOTA DE MORATORIA  -------------------

	IF(@pModoDebug = 1)
	BEGIN
		PRINT 'Genero ND en cero con los meses incluidos en la moratoria'
		PRINT 'Total de Items en ND: ' + CONVERT(VARCHAR(4),@Id_Grupo_Cont)
		PRINT 'Genero ND tipo: ' + @Tipo_ND
	END

	SET @JSON_Parametros_Stored_Invocado =	'{' +
													dbo.CF_Json_ParsearVariable('FlagUsarTrans', 0, 'TINYINT') + ',' +
													dbo.CF_Json_ParsearVariable('ModoDebug', @pModoDebug, 'TINYINT') + ',' +										
													dbo.CF_Json_ParsearVariable('Id_Usuario', -1, 'SMALLINT') + ',' +
													dbo.CF_Json_ParsearVariable('Omitir_Resultado_OK', 1, 'TINYINT') + ',' +
													dbo.CF_Json_ParsearVariable('Lote', '', 'VARCHAR(20)') + ',' +
													dbo.CF_Json_ParsearVariable('Fecha', GETDATE(), 'DATET') + ',' +
													dbo.CF_Json_ParsearVariable('FechaVence', GETDATE(), 'DATE') + ',' +
													dbo.CF_Json_ParsearVariable('Id_Cliente', @pId_Cliente, 'INT') + ',' +
													dbo.CF_Json_ParsearVariable('Id_Tipo_Cbte', @Tipo_ND, 'CHAR(4)') + ',' +
													dbo.CF_Json_ParsearVariable('Id_Cond_Pago', @Cond_Pago, 'SMALLINT') + ',' +
                                                    dbo.CF_Json_ParsearVariable('Id_Tarjeta',@pId_Tarjeta,'VARCHAR(10)') + ',' +
                                                    dbo.CF_Json_ParsearVariable('Tar_Nro',@pTar_Nro,'VARCHAR(50)') + ',' +
													dbo.CF_Json_ParsearVariable('Id_Moneda', 'PES', 'CHAR(3)') + ',' +
													dbo.CF_Json_ParsearVariable('Id_Lista_Precio', @Id_Lista_Precio_Cbtes, 'SMALLINT') + ',' +
													dbo.CF_Json_ParsearVariable('Id_Contrato', @Id_Contrato, 'INT') +
													'}'

	IF(@pModoDebug = 1)
	BEGIN
		PRINT 'Ingreso a Stored Procedure: CPJ_Cbtes_Fac_Generar'
		PRINT 'Parametros:'
		PRINT '@JSON_Parametros_Stored_Invocado: ' + @JSON_Parametros_Stored_Invocado
	END														
													
	EXEC @UltError = dbo.CPJ_Cbtes_Fac_Generar @JSON_Parametros_Stored_Invocado

	IF(@UltError <> 0)
	BEGIN
		RAISERROR ('Error when calling CPJ_Cbtes_Fac_Generar NC', 16, 1) 
	END

	SELECT @Id_Trans_ND = MAX(Id_Trans) FROM Cbtes WHERE Id_Contrato = @Id_Contrato AND Id_Tipo_Cbte = @Tipo_ND


    ----GENERO LOS ITEMS
	
	IF(@pModoDebug = 1)
	BEGIN
		PRINT 'Transaccion de NC generada: ' + CONVERT(VARCHAR(25),@Id_Trans_NC)
	END

	SET @Precio = 0

	IF(@pModoDebug = 1)
	BEGIN
		PRINT 'Genero item de Moratoria para NC por el total de ' + CONVERT(VARCHAR(30),@Precio)
	END

	SET @Mes = DATEADD(MONTH,1,@Ult_Cuota_Paga)
	WHILE (@Mes <  DATEADD(DAY,1,EOMONTH(GETDATE(),-1)))
	BEGIN

		SET @JSON_Parametros_Stored_Invocado =	'{' +
													dbo.CF_Json_ParsearVariable('FlagUsarTrans', 0, 'TINYINT') + ',' +
													dbo.CF_Json_ParsearVariable('ModoDebug', @pModoDebug, 'TINYINT') + ',' +										
													dbo.CF_Json_ParsearVariable('Id_Trans', @Id_Trans_ND, 'INT') + ',' +
													dbo.CF_Json_ParsearVariable('Omitir_Resultado_OK', 1, 'TINYINT') + ',' +
													dbo.CF_Json_ParsearVariable('Id_Producto', @Id_Producto_CS, 'VARCHAR(14)') + ',' +
													dbo.CF_Json_ParsearVariable('Cantidad', 1, 'DECIMAL(15,5)') + ',' +
													dbo.CF_Json_ParsearVariable('Fecha_QA', @Mes, 'DATETIME') + ',' +
													dbo.CF_Json_ParsearVariable('Fecha_Venc', @Mes, 'DATETIME') + ',' +
													dbo.CF_Json_ParsearVariable('Id_Cliente', @pId_Cliente, 'INT') + ',' +
													dbo.CF_Json_ParsearVariable('Lote', '', 'VARCHAR(20)') + ',' +
													dbo.CF_Json_ParsearVariable('Refacturar', 1, 'TINYINT') + ',' +
													dbo.CF_Json_ParsearVariable('Bonif', 0, 'DECIMAL(5,2)') + ',' +
													dbo.CF_Json_ParsearVariable('Precio_Grav', @Precio, 'DECIMAL(15,5)') + ',' +
													dbo.CF_Json_ParsearVariable('Cantidad_Equiv', 0, 'DECIMAL(15,5)') + ',' +
													dbo.CF_Json_ParsearVariable('Precio_Exen', 0, 'DECIMAL(15,5)') + 
												'}'

		IF(@pModoDebug = 1)
		BEGIN
			PRINT 'Ingreso a Stored Procedure: CPJ_Cbtes_Fac_Generar_Item'
			PRINT 'Parametros:'
			PRINT '@JSON_Parametros_Stored_Invocado: ' + @JSON_Parametros_Stored_Invocado
		END			

		EXEC @UltError = dbo.CPJ_Cbtes_Fac_Generar_Item @JSON_Parametros_Stored_Invocado

		IF(@UltError <> 0)
		BEGIN
			RAISERROR ('Error when calling CPJ_Cbtes_Fac_Generar_Item', 16, 1) 
		END

		SET @Mes = DATEADD(MONTH,1,@Mes)
    END 



	SET @JSON_Parametros_Stored_Invocado =	'{' +
												dbo.CF_Json_ParsearVariable('FlagUsarTrans', 0, 'TINYINT') + ',' +
												dbo.CF_Json_ParsearVariable('ModoDebug', @pModoDebug, 'TINYINT') + ',' +										
												dbo.CF_Json_ParsearVariable('Id_Trans', @Id_Trans_ND, 'INT') + ',' +
												dbo.CF_Json_ParsearVariable('Omitir_Resultado_OK', 1, 'TINYINT') + ',' +
												dbo.CF_Json_ParsearVariable('Id_Producto', @Id_Producto_Item, 'VARCHAR(14)') + ',' +
												dbo.CF_Json_ParsearVariable('Cantidad', 1, 'DECIMAL(15,5)') + ',' +
												dbo.CF_Json_ParsearVariable('Fecha_QA', dbo.CF_FOMONTH(GETDATE()), 'DATE') + ',' +
												dbo.CF_Json_ParsearVariable('Fecha_Venc', GETDATE(), 'DATE') + ',' +
												dbo.CF_Json_ParsearVariable('Id_Cliente', @pId_Cliente, 'INT') + ',' +
												dbo.CF_Json_ParsearVariable('Lote', '', 'VARCHAR(20)') + ',' +
												dbo.CF_Json_ParsearVariable('Refacturar', 1, 'TINYINT') + ',' +
												dbo.CF_Json_ParsearVariable('Bonif', 0, 'DECIMAL(5,2)') + ',' +
												dbo.CF_Json_ParsearVariable('Precio_Grav', @Valor_Cuota, 'DECIMAL(15,5)') + ',' +
												dbo.CF_Json_ParsearVariable('Cantidad_Equiv', 0, 'DECIMAL(15,5)') + ',' +
												dbo.CF_Json_ParsearVariable('Precio_Exen', 0, 'DECIMAL(15,5)') + 
											'}'

	IF(@pModoDebug = 1)
	BEGIN
		PRINT 'Ingreso a Stored Procedure: CPJ_Cbtes_Fac_Generar_Item'
		PRINT 'Parametros:'
		PRINT '@JSON_Parametros_Stored_Invocado: ' + @JSON_Parametros_Stored_Invocado
	END			

	EXEC @UltError = dbo.CPJ_Cbtes_Fac_Generar_Item @JSON_Parametros_Stored_Invocado

	IF(@UltError <> 0)
	BEGIN
		RAISERROR ('Error when calling CPJ_Cbtes_Fac_Generar_Item', 16, 1) 
	END


	SET @Precio	= dbo.CF_Produc_Precio(@Id_Producto_CS, @pId_Cliente, dbo.CF_FOMONTH(GETDATE()), 0, 0,  'B')
	SET @JSON_Parametros_Stored_Invocado =	'{' +
												dbo.CF_Json_ParsearVariable('FlagUsarTrans', 0, 'TINYINT') + ',' +
												dbo.CF_Json_ParsearVariable('ModoDebug', @pModoDebug, 'TINYINT') + ',' +										
												dbo.CF_Json_ParsearVariable('Id_Trans', @Id_Trans_ND, 'INT') + ',' +
												dbo.CF_Json_ParsearVariable('Omitir_Resultado_OK', 1, 'TINYINT') + ',' +
												dbo.CF_Json_ParsearVariable('Id_Producto', @Id_Producto_CS, 'VARCHAR(14)') + ',' +
												dbo.CF_Json_ParsearVariable('Cantidad', 1, 'DECIMAL(15,5)') + ',' +
												dbo.CF_Json_ParsearVariable('Fecha_QA', dbo.CF_FOMONTH(GETDATE()), 'DATE') + ',' +
												dbo.CF_Json_ParsearVariable('Fecha_Venc', dbo.CF_EOMONTH(GETDATE()), 'DATE') + ',' +
												dbo.CF_Json_ParsearVariable('Id_Cliente', @pId_Cliente, 'INT') + ',' +
												dbo.CF_Json_ParsearVariable('Lote', '', 'VARCHAR(20)') + ',' +
												dbo.CF_Json_ParsearVariable('Refacturar', 1, 'TINYINT') + ',' +
												dbo.CF_Json_ParsearVariable('Bonif', 0, 'DECIMAL(5,2)') + ',' +
												dbo.CF_Json_ParsearVariable('Precio_Grav', @Precio, 'DECIMAL(15,5)') + ',' +
												dbo.CF_Json_ParsearVariable('Cantidad_Equiv', 0, 'DECIMAL(15,5)') + ',' +
												dbo.CF_Json_ParsearVariable('Precio_Exen', 0, 'DECIMAL(15,5)') + 
											'}'

	IF(@pModoDebug = 1)
	BEGIN
		PRINT 'Ingreso a Stored Procedure: CPJ_Cbtes_Fac_Generar_Item'
		PRINT 'Parametros:'
		PRINT '@JSON_Parametros_Stored_Invocado: ' + @JSON_Parametros_Stored_Invocado
	END			

	EXEC @UltError = dbo.CPJ_Cbtes_Fac_Generar_Item @JSON_Parametros_Stored_Invocado

	IF(@UltError <> 0)
	BEGIN
		RAISERROR ('Error when calling CPJ_Cbtes_Fac_Generar_Item', 16, 1) 
	END




	IF(@pModoDebug = 1)
	BEGIN
		PRINT 'GENERO ND'
	END

	SET @JSON_Parametros_Stored_Invocado =	'{' +
												dbo.CF_Json_ParsearVariable('FlagUsarTrans', 0, 'TINYINT') + ',' +
												dbo.CF_Json_ParsearVariable('ModoDebug', @pModoDebug, 'TINYINT') + ',' +
												dbo.CF_Json_ParsearVariable('Omitir_Result_OK', 1, 'TINYINT') + ',' +
												dbo.CF_Json_ParsearVariable('Id_Trans', @Id_Trans_ND, 'INT') + 
											'}'
	
	IF(@pModoDebug = 1)
	BEGIN
		PRINT 'Ingreso a Stored Procedure: CPJ_Cbtes_Fac_Fin (3)'
		PRINT 'Parametros:'
		PRINT '@JSON_Parametros_Stored_Invocado: ' + @JSON_Parametros_Stored_Invocado
	END	


	EXEC @UltError = dbo.CPJ_Cbtes_Fac_Fin @JSON_Parametros_Stored_Invocado

	IF(@UltError <> 0)
	BEGIN
		RAISERROR ('Error when calling CPJ_Cbtes_Fac_Fin (3)', 16, 1) 
	END
 
    ------------------- FIN ND MESES INCLUIDOS --------------------------------------------------------


	---- GENERO CUOTAS MORATORIA

    
	SET @Precio = (SELECT dbo.CFUJ_Moratoria_Deuda(@JSON_Parametros_Stored_Invocado)) / @Cant_Cuotas_Finan
	
	SET @Items = 2
	WHILE(@Items <= @Cant_Cuotas_Finan)
	BEGIN
	    SET @Fecha_Vto = DATEADD(MONTH,@Items-1,dbo.CF_FOMONTH(GETDATE()))

		IF(@pModoDebug = 1)
		BEGIN
			PRINT 'Genero ND POR Cantidad cuotas moratoria'
			PRINT 'Total de Items en ND: ' + CONVERT(VARCHAR(4),@Id_Grupo_Cont)
			PRINT 'Genero ND tipo: ' + @Tipo_ND
		END

		SET @JSON_Parametros_Stored_Invocado =	'{' +
														dbo.CF_Json_ParsearVariable('FlagUsarTrans', 0, 'TINYINT') + ',' +
														dbo.CF_Json_ParsearVariable('ModoDebug', @pModoDebug, 'TINYINT') + ',' +										
														dbo.CF_Json_ParsearVariable('Id_Usuario', -1, 'SMALLINT') + ',' +
														dbo.CF_Json_ParsearVariable('Omitir_Resultado_OK', 1, 'TINYINT') + ',' +
														dbo.CF_Json_ParsearVariable('Lote', '', 'VARCHAR(20)') + ',' +
														dbo.CF_Json_ParsearVariable('Fecha', GETDATE(), 'DATETIME') + ',' +
														dbo.CF_Json_ParsearVariable('FechaVence', @FECHA_VTO, 'DATETIME') + ',' +
														dbo.CF_Json_ParsearVariable('Id_Cliente', @pId_Cliente, 'INT') + ',' +
														dbo.CF_Json_ParsearVariable('Id_Tipo_Cbte', @Tipo_ND, 'CHAR(4)') + ',' +
														dbo.CF_Json_ParsearVariable('Id_Cond_Pago', @Cond_Pago, 'SMALLINT') + ',' +
                                                        dbo.CF_Json_ParsearVariable('Id_Tarjeta',@pId_Tarjeta,'VARCHAR(10)') + ',' +
                                                        dbo.CF_Json_ParsearVariable('Tar_Nro',@pTar_Nro,'VARCHAR(50)') + ',' +
														dbo.CF_Json_ParsearVariable('Id_Moneda', 'PES', 'CHAR(3)') + ',' +
														dbo.CF_Json_ParsearVariable('Id_Lista_Precio', @Id_Lista_Precio_Cbtes, 'SMALLINT') + ',' +
														dbo.CF_Json_ParsearVariable('Id_Contrato', @Id_Contrato, 'INT') +
														'}'

		IF(@pModoDebug = 1)
		BEGIN
			PRINT 'Ingreso a Stored Procedure: CPJ_Cbtes_Fac_Generar'
			PRINT 'Parametros:'
			PRINT '@JSON_Parametros_Stored_Invocado: ' + @JSON_Parametros_Stored_Invocado
		END														
													
		EXEC @UltError = dbo.CPJ_Cbtes_Fac_Generar @JSON_Parametros_Stored_Invocado

		IF(@UltError <> 0)
		BEGIN
			RAISERROR ('Error when calling CPJ_Cbtes_Fac_Generar ND moratoria', 16, 1) 
		END

		SELECT @Id_Trans_ND = MAX(Id_Trans) FROM Cbtes WHERE Id_Contrato = @Id_Contrato AND Id_Tipo_Cbte = @Tipo_ND

		----GENERO LOS ITEMS
	
		IF(@pModoDebug = 1)
		BEGIN
			PRINT 'Transaccion de ND generada: ' + CONVERT(VARCHAR(25),@Id_Trans_ND)
			PRINT 'Genero item de Moratoria para ND por el total de ' + CONVERT(VARCHAR(30),@Precio)
		END


		SET @JSON_Parametros_Stored_Invocado =	'{' +
													dbo.CF_Json_ParsearVariable('FlagUsarTrans', 0, 'TINYINT') + ',' +
													dbo.CF_Json_ParsearVariable('ModoDebug', @pModoDebug, 'TINYINT') + ',' +										
													dbo.CF_Json_ParsearVariable('Id_Trans', @Id_Trans_ND, 'INT') + ',' +
													dbo.CF_Json_ParsearVariable('Omitir_Resultado_OK', 1, 'TINYINT') + ',' +
													dbo.CF_Json_ParsearVariable('Id_Producto', @Id_Producto_Item, 'VARCHAR(14)') + ',' +
													dbo.CF_Json_ParsearVariable('Cantidad', 1, 'DECIMAL(15,5)') + ',' +
													dbo.CF_Json_ParsearVariable('Fecha_QA', @FECHA_VTO, 'DATETIME') + ',' +
													dbo.CF_Json_ParsearVariable('Fecha_Venc', @FECHA_VTO, 'DATETIME') + ',' +
													dbo.CF_Json_ParsearVariable('Id_Cliente', @pId_Cliente, 'INT') + ',' +
													dbo.CF_Json_ParsearVariable('Lote', '', 'VARCHAR(20)') + ',' +
													dbo.CF_Json_ParsearVariable('Refacturar', 1, 'TINYINT') + ',' +
													dbo.CF_Json_ParsearVariable('Bonif', 0, 'DECIMAL(5,2)') + ',' +
													dbo.CF_Json_ParsearVariable('Precio_Grav', @Valor_Cuota, 'DECIMAL(15,5)') + ',' +
													dbo.CF_Json_ParsearVariable('Cantidad_Equiv', 0, 'DECIMAL(15,5)') + ',' +
													dbo.CF_Json_ParsearVariable('Precio_Exen', 0, 'DECIMAL(15,5)') + 
												'}'

		IF(@pModoDebug = 1)
		BEGIN
			PRINT 'Ingreso a Stored Procedure: CPJ_Cbtes_Fac_Generar_Item'
			PRINT 'Parametros:'
			PRINT '@JSON_Parametros_Stored_Invocado: ' + @JSON_Parametros_Stored_Invocado
		END			

		EXEC @UltError = dbo.CPJ_Cbtes_Fac_Generar_Item @JSON_Parametros_Stored_Invocado

		IF(@UltError <> 0)
		BEGIN
			RAISERROR ('Error when calling CPJ_Cbtes_Fac_Generar_Item', 16, 1) 
		END

		IF(@pModoDebug = 1)
		BEGIN
			PRINT 'Genero ND'
		END

		SET @JSON_Parametros_Stored_Invocado =	'{' +
													dbo.CF_Json_ParsearVariable('FlagUsarTrans', 0, 'TINYINT') + ',' +
													dbo.CF_Json_ParsearVariable('ModoDebug', @pModoDebug, 'TINYINT') + ',' +
													dbo.CF_Json_ParsearVariable('Omitir_Result_OK', 1, 'TINYINT') + ',' +
													dbo.CF_Json_ParsearVariable('Id_Trans', @Id_Trans_ND, 'INT') + 
												'}'
	
		IF(@pModoDebug = 1)
		BEGIN
			PRINT 'Ingreso a Stored Procedure: CPJ_Cbtes_Fac_Fin (3)'
			PRINT 'Parametros:'
			PRINT '@JSON_Parametros_Stored_Invocado: ' + @JSON_Parametros_Stored_Invocado
		END	

		EXEC @UltError = dbo.CPJ_Cbtes_Fac_Fin @JSON_Parametros_Stored_Invocado

		IF(@UltError <> 0)
		BEGIN
			RAISERROR ('Error when calling CPJ_Cbtes_Fac_Fin (3)', 16, 1) 
		END

		SET @Items += 1
	END

	SET @Id_Trans_Generada = (SELECT MIN(C.Id_Trans) FROM Cbtes C, Cbtes_Tipos T 
	       WHERE C.Id_Contrato = @Id_Contrato 
		   AND C.Id_Tipo_Cbte = T.Id_Tipo_Cbte 
		   AND T.Compromete_Factura = 1
		   AND C.Id_Estado_Cbte = 1
		   )
	
	--RAISERROR ('prueba trans', 16, 1) 

	IF @pFlagUsarTrans <> 0 AND @@TRANCOUNT > 0
	BEGIN
		COMMIT
	END

	IF @pOmitir_Resultado_OK = 0
	BEGIN
		SET @rtaMsjStored	=	@Id_Trans_Generada

		SELECT	 0				AS ErrorNumber 
				,0				AS ErrorSeverity  
				,0				AS ErrorState 
				,0				AS ErrorProcedure  
				,@Id_Contrato	AS ErrorLine		
				,@rtaMsjStored	AS ErrorMessage
	END

	IF(@pModoDebug = 1)
	BEGIN
		PRINT 'Fin Stored Procedure: CPJ_Moratoria_Generar'
	END
	
END TRY
BEGIN CATCH

	IF  (SELECT COUNT(*) FROM @Respuesta_ WHERE [ErrorNumber] > 0 ) > 0
	BEGIN
		SELECT * FROM @Respuesta_
	END
	ELSE
	BEGIN
		EXECUTE usp_GetErrorInfo
	END

	IF @pFlagUsarTrans <> 0 AND @@TRANCOUNT > 0
	BEGIN
		ROLLBACK
	END

	RETURN 9999

END CATCH
